const { PlayerManager, RoomManager } = require('./objects');
const { generateId, shuffleArray, getQuiplashPrompts, getTriviaQuestions } = require('./utils');

const players = new PlayerManager();
const roomManager = new RoomManager();

// Game phase constants
const PHASES = {
  LOBBY: 'lobby',
  PROMPTS: 'prompts',
  VOTING: 'voting',
  RESULTS: 'results',
  FINAL_SCORES: 'finalScores',
  GAME_OVER: 'gameOver'
};

function attachListeners(io, gameReference) {
  io.on('connect', socket => {
    console.log('Client connected:', socket.id);

    /**
     * Room events
     */
    socket.on('hb-createRoom', () => {
      const id = generateId();
      socket.join(id);
      const room = roomManager.addRoom(id, socket.id, 8);
      room.phase = PHASES.LOBBY;
      room.gameType = null;
      room.currentRound = 0;
      room.totalRounds = 3;
      room.prompts = [];
      room.currentPromptIndex = 0;
      room.answers = {};
      room.votes = {};
      room.triviaAnswers = {};
      io.to(id).emit('hb-roomData', room);
      console.log('Room created:', id);
    });

    socket.on('hb-selectGameType', ({ roomId, gameType }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;
      
      room.gameType = gameType;
      room.gameConfig = gameReference[gameType];
      
      // Notify host and all players
      io.to(room.socketId).emit('hb-gameTypeSelected', { gameType, config: room.gameConfig });
      room.players.forEach(player => {
        io.to(player.socketId).emit('hb-gameTypeSelected', { gameType, config: room.gameConfig });
      });
    });

    socket.on('hb-startGame', ({ roomId, gameType }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      room.gameType = gameType;
      room.gameConfig = gameReference[gameType];
      room.currentRound = 1;
      room.phase = PHASES.PROMPTS;

      // Reset scores
      room.players.forEach(player => {
        player.score = 0;
      });

      if (gameType === 'quiplash') {
        startQuiplashRound(io, room, gameReference);
      } else if (gameType === 'trivia') {
        startTriviaRound(io, room, gameReference);
      } else if (gameType === 'drawful') {
        startDrawfulRound(io, room, gameReference);
      }

      // Notify everyone
      io.to(room.socketId).emit('hb-gameStart', { gameType, room });
      room.players.forEach(player => {
        io.to(player.socketId).emit('hb-gameStart', { gameType, room });
      });
    });

    /**
     * Player events
     */
    socket.on('hb-joinRoom', ({ roomId, playerName }) => {
      const roomIdUpper = roomId.toUpperCase();
      
      if (roomManager.roomExists(roomIdUpper) === false) {
        io.to(socket.id).emit('hb-error', 'Room not found');
        return;
      }

      const room = roomManager.getRoom(roomIdUpper);
      if (room.players.length >= room.maxPlayers) {
        io.to(socket.id).emit('hb-error', 'Room is full');
        return;
      }

      // Check for duplicate names
      const existingPlayer = room.players.find(p => p.name.toLowerCase() === playerName.toLowerCase());
      if (existingPlayer) {
        io.to(socket.id).emit('hb-error', 'Name already taken');
        return;
      }

      const playerId = generateId();
      const newPlayer = players.addPlayer({ 
        id: playerId, 
        roomId: roomIdUpper, 
        socketId: socket.id, 
        name: playerName,
        color: getRandomColor()
      });
      roomManager.addPlayer(roomIdUpper, newPlayer);

      socket.join(roomIdUpper);

      io.to(room.socketId).emit('hb-onPlayerJoin', room);
      io.to(socket.id).emit('hb-roomConnectionSuccessful', { playerId, room });
      
      console.log(`${playerName} joined room ${roomIdUpper}`);
    });

    socket.on('hb-leaveRoom', ({ roomId, playerId }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      roomManager.removePlayer(roomId, playerId);
      players.removePlayer(playerId);
      
      socket.leave(roomId);
      io.to(room.socketId).emit('hb-onPlayerLeave', room);
    });

    /**
     * Quiplash events
     */
    socket.on('hb-submitAnswer', ({ roomId, playerId, promptId, answer }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      if (!room.answers[promptId]) {
        room.answers[promptId] = [];
      }

      // Check if player already answered
      const existingAnswer = room.answers[promptId].find(a => a.playerId === playerId);
      if (existingAnswer) {
        existingAnswer.answer = answer;
      } else {
        room.answers[promptId].push({ playerId, answer, votes: 0 });
      }

      const player = room.players.find(p => p.id === playerId);
      if (player) {
        player.hasAnswered = true;
      }

      // Check if all required answers are in
      const totalAnswersNeeded = room.currentPrompts ? room.currentPrompts.length * 2 : 0;
      const totalAnswers = Object.values(room.answers).reduce((sum, arr) => sum + arr.length, 0);

      io.to(room.socketId).emit('hb-answerSubmitted', { 
        playerId, 
        promptId,
        answersReceived: totalAnswers,
        answersNeeded: totalAnswersNeeded,
        room 
      });

      // Check if all players have answered their prompts
      const allAnswered = checkAllAnswered(room);
      if (allAnswered) {
        startVotingPhase(io, room);
      }
    });

    socket.on('hb-submitVote', ({ roomId, playerId, promptId, votedAnswerPlayerId }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      if (!room.votes[promptId]) {
        room.votes[promptId] = [];
      }

      // Check if player already voted
      const existingVote = room.votes[promptId].find(v => v.visiblePlayerId === playerId);
      if (!existingVote) {
        room.votes[promptId].push({ visiblePlayerId: playerId, votedFor: votedAnswerPlayerId });
        
        // Add vote to the answer
        const answers = room.answers[promptId];
        if (answers) {
          const votedAnswer = answers.find(a => a.playerId === votedAnswerPlayerId);
          if (votedAnswer) {
            votedAnswer.votes++;
          }
        }
      }

      // Check if all non-answering players have voted
      const answererIds = room.answers[promptId]?.map(a => a.playerId) || [];
      const eligibleVoters = room.players.filter(p => !answererIds.includes(p.id));
      const votesReceived = room.votes[promptId]?.length || 0;

      io.to(room.socketId).emit('hb-voteSubmitted', { 
        promptId, 
        votesReceived,
        votesNeeded: eligibleVoters.length,
        room 
      });

      if (votesReceived >= eligibleVoters.length) {
        showPromptResults(io, room, promptId);
      }
    });

    socket.on('hb-nextPrompt', ({ roomId }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      room.currentPromptIndex++;
      
      if (room.currentPromptIndex >= room.currentPrompts.length) {
        // All prompts done, check for next round or end game
        if (room.currentRound >= room.totalRounds) {
          endGame(io, room);
        } else {
          room.currentRound++;
          startQuiplashRound(io, room);
        }
      } else {
        // Show next prompt for voting
        const currentPrompt = room.currentPrompts[room.currentPromptIndex];
        const answers = room.answers[currentPrompt.id] || [];
        
        room.phase = PHASES.VOTING;
        
        io.to(room.socketId).emit('hb-showVoting', { prompt: currentPrompt, answers, room });
        room.players.forEach(player => {
          const isAnswerer = answers.some(a => a.playerId === player.id);
          io.to(player.socketId).emit('hb-showVoting', { 
            prompt: currentPrompt, 
            answers: isAnswerer ? [] : answers, // Don't show own answers
            canVote: !isAnswerer,
            room 
          });
        });
      }
    });

    /**
     * Trivia events
     */
    socket.on('hb-submitTriviaAnswer', ({ roomId, playerId, questionId, answerIndex, timeRemaining }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      if (!room.triviaAnswers[questionId]) {
        room.triviaAnswers[questionId] = [];
      }

      const existingAnswer = room.triviaAnswers[questionId].find(a => a.playerId === playerId);
      if (!existingAnswer) {
        room.triviaAnswers[questionId].push({ 
          playerId, 
          answerIndex, 
          timeRemaining,
          timestamp: Date.now()
        });
      }

      io.to(room.socketId).emit('hb-triviaAnswerSubmitted', { 
        playerId, 
        answersReceived: room.triviaAnswers[questionId].length,
        totalPlayers: room.players.length
      });

      // Check if all players answered
      if (room.triviaAnswers[questionId].length >= room.players.length) {
        showTriviaResults(io, room, questionId);
      }
    });

    socket.on('hb-nextQuestion', ({ roomId }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      room.currentQuestionIndex++;
      
      if (room.currentQuestionIndex >= room.questions.length) {
        if (room.currentRound >= room.totalRounds) {
          endGame(io, room);
        } else {
          room.currentRound++;
          startTriviaRound(io, room);
        }
      } else {
        sendNextTriviaQuestion(io, room);
      }
    });

    /**
     * Drawful events
     */
    socket.on('hb-submitDrawing', ({ roomId, playerId, promptId, drawingData }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      if (!room.drawings) room.drawings = {};
      room.drawings[promptId] = { playerId, drawingData, fakeAnswers: [], votes: {} };

      const player = room.players.find(p => p.id === playerId);
      if (player) player.hasDrawn = true;

      io.to(room.socketId).emit('hb-drawingSubmitted', { playerId, room });

      // Check if all drawings are in
      const allDrawn = room.players.every(p => p.hasDrawn);
      if (allDrawn) {
        startDrawfulGuessing(io, room);
      }
    });

    socket.on('hb-submitFakeAnswer', ({ roomId, playerId, drawingId, fakeAnswer }) => {
      const room = roomManager.getRoom(roomId);
      if (!room || !room.drawings[drawingId]) return;

      room.drawings[drawingId].fakeAnswers.push({ playerId, answer: fakeAnswer });

      // Check if all fake answers are in
      const artistId = room.drawings[drawingId].playerId;
      const guessers = room.players.filter(p => p.id !== artistId);
      
      if (room.drawings[drawingId].fakeAnswers.length >= guessers.length) {
        startDrawfulVoting(io, room, drawingId);
      }
    });

    /**
     * General game events
     */
    socket.on('hb-forceNextPhase', ({ roomId }) => {
      const room = roomManager.getRoom(roomId);
      if (!room) return;

      if (room.phase === PHASES.PROMPTS) {
        startVotingPhase(io, room);
      } else if (room.phase === PHASES.VOTING) {
        room.currentPromptIndex++;
        if (room.currentPromptIndex >= room.currentPrompts.length) {
          if (room.currentRound >= room.totalRounds) {
            endGame(io, room);
          } else {
            room.currentRound++;
            if (room.gameType === 'quiplash') {
              startQuiplashRound(io, room);
            } else if (room.gameType === 'trivia') {
              startTriviaRound(io, room);
            }
          }
        }
      }
    });

    socket.on('hb-getRoom', ({ roomId }) => {
      const room = roomManager.getRoom(roomId);
      if (room) {
        io.to(socket.id).emit('hb-roomData', room);
      }
    });

    socket.on('hb-playerStatusUpdate', data => {
      const room = roomManager.getRoom(data.id);
      if (!room) return;

      roomManager.updatePlayerStatus(data.id, data.playerId, data.playerIsReady);

      if (roomManager.allReady(data.id)) {
        io.to(room.socketId).emit('hb-playersReady', true);
      } else {
        io.to(room.socketId).emit('hb-playersReady', false);
      }

      io.to(room.socketId).emit('hb-updateData', room);
    });

    socket.on('disconnect', () => {
      console.log('Client disconnected:', socket.id);
      // Handle player disconnect - find and remove from room
      const player = players.getPlayerBySocketId(socket.id);
      if (player) {
        const room = roomManager.getRoom(player.roomId);
        if (room) {
          roomManager.removePlayer(player.roomId, player.id);
          players.removePlayer(player.id);
          io.to(room.socketId).emit('hb-onPlayerLeave', room);
        }
      }
    });
  });
}

// Helper functions
function getRandomColor() {
  const colors = [
    '#FF6B6B', '#4ECDC4', '#45B7D1', '#96CEB4', 
    '#FFEAA7', '#DDA0DD', '#98D8C8', '#F7DC6F',
    '#BB8FCE', '#85C1E9', '#F8B500', '#00CED1'
  ];
  return colors[Math.floor(Math.random() * colors.length)];
}

function checkAllAnswered(room) {
  if (!room.currentPrompts) return false;
  
  for (const prompt of room.currentPrompts) {
    const answers = room.answers[prompt.id] || [];
    if (answers.length < 2) return false;
  }
  return true;
}

function startQuiplashRound(io, room, gameReference) {
  room.phase = PHASES.PROMPTS;
  room.answers = {};
  room.votes = {};
  room.currentPromptIndex = 0;

  // Get prompts for this round
  const prompts = getQuiplashPrompts(room.players.length);
  room.currentPrompts = prompts;

  // Assign 2 prompts to each player
  const playerPrompts = {};
  const shuffledPlayers = shuffleArray([...room.players]);
  
  prompts.forEach((prompt, index) => {
    const player1 = shuffledPlayers[index % shuffledPlayers.length];
    const player2 = shuffledPlayers[(index + 1) % shuffledPlayers.length];
    
    prompt.assignedPlayers = [player1.id, player2.id];
    
    if (!playerPrompts[player1.id]) playerPrompts[player1.id] = [];
    if (!playerPrompts[player2.id]) playerPrompts[player2.id] = [];
    
    playerPrompts[player1.id].push(prompt);
    playerPrompts[player2.id].push(prompt);
  });

  // Send prompts to each player
  room.players.forEach(player => {
    player.hasAnswered = false;
    const myPrompts = playerPrompts[player.id] || [];
    io.to(player.socketId).emit('hb-receivePrompts', { 
      prompts: myPrompts, 
      round: room.currentRound,
      totalRounds: room.totalRounds
    });
  });

  io.to(room.socketId).emit('hb-roundStart', { 
    round: room.currentRound, 
    phase: PHASES.PROMPTS,
    room 
  });
}

function startVotingPhase(io, room) {
  room.phase = PHASES.VOTING;
  room.currentPromptIndex = 0;

  if (room.currentPrompts && room.currentPrompts.length > 0) {
    const currentPrompt = room.currentPrompts[0];
    const answers = room.answers[currentPrompt.id] || [];

    io.to(room.socketId).emit('hb-showVoting', { prompt: currentPrompt, answers, room });
    
    room.players.forEach(player => {
      const isAnswerer = answers.some(a => a.playerId === player.id);
      io.to(player.socketId).emit('hb-showVoting', { 
        prompt: currentPrompt, 
        answers: isAnswerer ? [] : answers,
        canVote: !isAnswerer,
        room 
      });
    });
  }
}

function showPromptResults(io, room, promptId) {
  room.phase = PHASES.RESULTS;
  
  const prompt = room.currentPrompts.find(p => p.id === promptId);
  const answers = room.answers[promptId] || [];
  
  // Calculate and award points
  answers.forEach(answer => {
    const player = room.players.find(p => p.id === answer.playerId);
    if (player) {
      const points = answer.votes * 100;
      player.score += points;
      answer.pointsAwarded = points;
      answer.playerName = player.name;
      answer.playerColor = player.color;
    }
  });

  // Check for Quiplash (one answer gets all votes)
  const totalVotes = answers.reduce((sum, a) => sum + a.votes, 0);
  const quiplash = answers.some(a => a.votes === totalVotes && totalVotes > 0);

  io.to(room.socketId).emit('hb-promptResults', { 
    prompt, 
    answers, 
    quiplash,
    room 
  });
  
  room.players.forEach(player => {
    io.to(player.socketId).emit('hb-promptResults', { 
      prompt, 
      answers, 
      quiplash,
      room 
    });
  });
}

function startTriviaRound(io, room, gameReference) {
  room.phase = PHASES.PROMPTS;
  room.triviaAnswers = {};
  room.currentQuestionIndex = 0;

  // Get trivia questions
  room.questions = getTriviaQuestions(5);

  io.to(room.socketId).emit('hb-roundStart', { 
    round: room.currentRound, 
    phase: PHASES.PROMPTS,
    totalQuestions: room.questions.length,
    room 
  });

  room.players.forEach(player => {
    io.to(player.socketId).emit('hb-roundStart', { 
      round: room.currentRound, 
      phase: PHASES.PROMPTS,
      room 
    });
  });

  // Send first question after a delay
  setTimeout(() => {
    sendNextTriviaQuestion(io, room);
  }, 3000);
}

function sendNextTriviaQuestion(io, room) {
  const question = room.questions[room.currentQuestionIndex];
  
  io.to(room.socketId).emit('hb-triviaQuestion', { 
    question, 
    questionNumber: room.currentQuestionIndex + 1,
    totalQuestions: room.questions.length,
    room 
  });
  
  room.players.forEach(player => {
    io.to(player.socketId).emit('hb-triviaQuestion', { 
      question, 
      questionNumber: room.currentQuestionIndex + 1,
      totalQuestions: room.questions.length
    });
  });

  // Auto-advance after time limit
  setTimeout(() => {
    if (room.phase !== PHASES.GAME_OVER) {
      showTriviaResults(io, room, question.id);
    }
  }, 15000);
}

function showTriviaResults(io, room, questionId) {
  room.phase = PHASES.RESULTS;
  
  const question = room.questions.find(q => q.id === questionId);
  const answers = room.triviaAnswers[questionId] || [];
  
  const results = room.players.map(player => {
    const answer = answers.find(a => a.playerId === player.id);
    const isCorrect = answer && answer.answerIndex === question.correctIndex;
    const points = isCorrect ? Math.floor(100 + (answer.timeRemaining * 10)) : 0;
    
    if (isCorrect) {
      player.score += points;
    }
    
    return {
      playerId: player.id,
      playerName: player.name,
      playerColor: player.color,
      answered: !!answer,
      answerIndex: answer?.answerIndex,
      isCorrect,
      points,
      totalScore: player.score
    };
  });

  io.to(room.socketId).emit('hb-triviaResults', { 
    question, 
    results,
    room 
  });
  
  room.players.forEach(player => {
    io.to(player.socketId).emit('hb-triviaResults', { 
      question, 
      results
    });
  });
}

function startDrawfulRound(io, room, gameReference) {
  room.phase = PHASES.PROMPTS;
  room.drawings = {};
  room.currentDrawingIndex = 0;

  // Assign one drawing prompt to each player
  const drawingPrompts = getDrawfulPrompts(room.players.length);
  
  room.players.forEach((player, index) => {
    player.hasDrawn = false;
    const prompt = drawingPrompts[index];
    io.to(player.socketId).emit('hb-drawingPrompt', { prompt });
  });

  io.to(room.socketId).emit('hb-roundStart', { 
    round: room.currentRound, 
    phase: PHASES.PROMPTS,
    room 
  });
}

function startDrawfulGuessing(io, room) {
  room.phase = PHASES.VOTING;
  const drawingIds = Object.keys(room.drawings);
  room.currentDrawingId = drawingIds[0];
  
  const drawing = room.drawings[room.currentDrawingId];
  
  // Ask non-artists to submit fake answers
  room.players.forEach(player => {
    if (player.id !== drawing.playerId) {
      io.to(player.socketId).emit('hb-submitFakeAnswer', { 
        drawingData: drawing.drawingData 
      });
    }
  });
}

function startDrawfulVoting(io, room, drawingId) {
  const drawing = room.drawings[drawingId];
  const allAnswers = [
    { answer: drawing.realAnswer, isReal: true },
    ...drawing.fakeAnswers.map(f => ({ answer: f.answer, playerId: f.playerId, isReal: false }))
  ];
  
  const shuffledAnswers = shuffleArray(allAnswers);
  
  room.players.forEach(player => {
    if (player.id !== drawing.playerId) {
      io.to(player.socketId).emit('hb-drawfulVote', { 
        drawingData: drawing.drawingData,
        answers: shuffledAnswers 
      });
    }
  });
  
  io.to(room.socketId).emit('hb-drawfulVoting', { 
    drawingData: drawing.drawingData,
    answers: shuffledAnswers,
    room 
  });
}

function getDrawfulPrompts(count) {
  const prompts = [
    'A cat wearing a top hat',
    'Pizza delivery on the moon',
    'A robot learning to dance',
    'Fish riding a bicycle',
    'A superhero doing laundry',
    'Dragons at a tea party',
    'Aliens playing basketball',
    'A penguin surfing',
    'A wizard making breakfast',
    'Dinosaurs at a disco'
  ];
  return shuffleArray(prompts).slice(0, count).map((text, i) => ({ id: generateId(), text }));
}

function endGame(io, room) {
  room.phase = PHASES.FINAL_SCORES;
  
  // Sort players by score
  const sortedPlayers = [...room.players].sort((a, b) => b.score - a.score);
  
  io.to(room.socketId).emit('hb-gameOver', { 
    finalScores: sortedPlayers,
    winner: sortedPlayers[0],
    room 
  });
  
  room.players.forEach(player => {
    io.to(player.socketId).emit('hb-gameOver', { 
      finalScores: sortedPlayers,
      winner: sortedPlayers[0]
    });
  });
}

module.exports = attachListeners;
