/**
 * Hackbox Client - For use in React/frontend applications
 * Include this in your client-side build
 */

const io = require('socket.io-client');

const hackboxClient = async url => {
  const socket = await io.connect(url);

  /**
   * Room/Host methods
   */
  const createRoom = () => {
    return new Promise((resolve, reject) => {
      socket.emit('hb-createRoom');
      socket.on('hb-roomData', data => {
        resolve(data);
      });
      setTimeout(() => reject(new Error('Timeout creating room')), 10000);
    });
  };

  const onPlayerJoin = cb => {
    socket.on('hb-onPlayerJoin', room => {
      cb(room);
    });
  };

  const onPlayerLeave = cb => {
    socket.on('hb-onPlayerLeave', room => {
      cb(room);
    });
  };

  const selectGameType = ({ roomId, gameType }) => {
    socket.emit('hb-selectGameType', { roomId, gameType });
  };

  const startGame = ({ roomId, gameType }) => {
    socket.emit('hb-startGame', { roomId, gameType });
  };

  const nextPrompt = ({ roomId }) => {
    socket.emit('hb-nextPrompt', { roomId });
  };

  const nextQuestion = ({ roomId }) => {
    socket.emit('hb-nextQuestion', { roomId });
  };

  const forceNextPhase = ({ roomId }) => {
    socket.emit('hb-forceNextPhase', { roomId });
  };

  /**
   * Player methods
   */
  const joinRoom = ({ roomId, playerName }) => {
    return new Promise((resolve, reject) => {
      socket.emit('hb-joinRoom', { roomId, playerName });
      
      socket.on('hb-roomConnectionSuccessful', data => {
        resolve(data);
      });
      
      socket.on('hb-error', error => {
        reject(new Error(error));
      });
      
      setTimeout(() => reject(new Error('Timeout joining room')), 10000);
    });
  };

  const leaveRoom = ({ roomId, playerId }) => {
    socket.emit('hb-leaveRoom', { roomId, playerId });
  };

  const submitAnswer = ({ roomId, playerId, promptId, answer }) => {
    socket.emit('hb-submitAnswer', { roomId, playerId, promptId, answer });
  };

  const submitVote = ({ roomId, playerId, promptId, votedAnswerPlayerId }) => {
    socket.emit('hb-submitVote', { roomId, playerId, promptId, votedAnswerPlayerId });
  };

  const submitTriviaAnswer = ({ roomId, playerId, questionId, answerIndex, timeRemaining }) => {
    socket.emit('hb-submitTriviaAnswer', { roomId, playerId, questionId, answerIndex, timeRemaining });
  };

  const submitDrawing = ({ roomId, playerId, promptId, drawingData }) => {
    socket.emit('hb-submitDrawing', { roomId, playerId, promptId, drawingData });
  };

  const submitFakeAnswer = ({ roomId, playerId, drawingId, fakeAnswer }) => {
    socket.emit('hb-submitFakeAnswer', { roomId, playerId, drawingId, fakeAnswer });
  };

  const updateStatus = ({ roomId, playerId, isReady }) => {
    socket.emit('hb-playerStatusUpdate', { id: roomId, playerId, playerIsReady: isReady });
  };

  const getRoom = ({ roomId }) => {
    return new Promise(resolve => {
      socket.emit('hb-getRoom', { roomId });
      socket.once('hb-roomData', data => {
        resolve(data);
      });
    });
  };

  /**
   * Event listeners
   */
  const onStartGame = cb => {
    socket.on('hb-gameStart', data => {
      cb(data);
    });
  };

  const onGameTypeSelected = cb => {
    socket.on('hb-gameTypeSelected', data => {
      cb(data);
    });
  };

  const onRoundStart = cb => {
    socket.on('hb-roundStart', data => {
      cb(data);
    });
  };

  const onReceivePrompts = cb => {
    socket.on('hb-receivePrompts', data => {
      cb(data);
    });
  };

  const onAnswerSubmitted = cb => {
    socket.on('hb-answerSubmitted', data => {
      cb(data);
    });
  };

  const onShowVoting = cb => {
    socket.on('hb-showVoting', data => {
      cb(data);
    });
  };

  const onVoteSubmitted = cb => {
    socket.on('hb-voteSubmitted', data => {
      cb(data);
    });
  };

  const onPromptResults = cb => {
    socket.on('hb-promptResults', data => {
      cb(data);
    });
  };

  const onTriviaQuestion = cb => {
    socket.on('hb-triviaQuestion', data => {
      cb(data);
    });
  };

  const onTriviaAnswerSubmitted = cb => {
    socket.on('hb-triviaAnswerSubmitted', data => {
      cb(data);
    });
  };

  const onTriviaResults = cb => {
    socket.on('hb-triviaResults', data => {
      cb(data);
    });
  };

  const onDrawingPrompt = cb => {
    socket.on('hb-drawingPrompt', data => {
      cb(data);
    });
  };

  const onDrawingSubmitted = cb => {
    socket.on('hb-drawingSubmitted', data => {
      cb(data);
    });
  };

  const onSubmitFakeAnswer = cb => {
    socket.on('hb-submitFakeAnswer', data => {
      cb(data);
    });
  };

  const onDrawfulVote = cb => {
    socket.on('hb-drawfulVote', data => {
      cb(data);
    });
  };

  const onDrawfulVoting = cb => {
    socket.on('hb-drawfulVoting', data => {
      cb(data);
    });
  };

  const onGameOver = cb => {
    socket.on('hb-gameOver', data => {
      cb(data);
    });
  };

  const onUpdateData = cb => {
    socket.on('hb-updateData', data => {
      cb(data);
    });
  };

  const onPlayersReady = cb => {
    socket.on('hb-playersReady', data => {
      cb(data);
    });
  };

  const onError = cb => {
    socket.on('hb-error', error => {
      cb(error);
    });
  };

  const disconnect = () => {
    socket.disconnect();
  };

  return {
    // Room/Host methods
    createRoom,
    onPlayerJoin,
    onPlayerLeave,
    selectGameType,
    startGame,
    nextPrompt,
    nextQuestion,
    forceNextPhase,
    
    // Player methods
    joinRoom,
    leaveRoom,
    submitAnswer,
    submitVote,
    submitTriviaAnswer,
    submitDrawing,
    submitFakeAnswer,
    updateStatus,
    getRoom,
    
    // Event listeners
    onStartGame,
    onGameTypeSelected,
    onRoundStart,
    onReceivePrompts,
    onAnswerSubmitted,
    onShowVoting,
    onVoteSubmitted,
    onPromptResults,
    onTriviaQuestion,
    onTriviaAnswerSubmitted,
    onTriviaResults,
    onDrawingPrompt,
    onDrawingSubmitted,
    onSubmitFakeAnswer,
    onDrawfulVote,
    onDrawfulVoting,
    onGameOver,
    onUpdateData,
    onPlayersReady,
    onError,
    disconnect
  };
};

module.exports = hackboxClient;
