class RoomManager {
  constructor () {
    this.rooms = [];
  }

  addRoom (id, socketId, maxPlayers) {
    const room = { 
      id, 
      socketId, 
      maxPlayers, 
      players: [],
      phase: 'lobby',
      gameType: null,
      gameConfig: null,
      currentRound: 0,
      totalRounds: 3,
      prompts: [],
      currentPrompts: [],
      currentPromptIndex: 0,
      answers: {},
      votes: {},
      questions: [],
      currentQuestionIndex: 0,
      triviaAnswers: {},
      drawings: {},
      currentDrawingIndex: 0,
      createdAt: Date.now()
    };
    this.rooms.push(room);
    return room;
  }

  removeRoom (id) {
    const removedRoom = this.rooms.find(room => room.id === id);

    if (removedRoom) {
      this.rooms = this.rooms.filter(room => room.id !== id);
    }

    return removedRoom;
  }

  getRoom (id) {
    return this.rooms.find(room => room.id === id);
  }

  addPlayer (roomId, player) {
    const room = this.getRoom(roomId);
    if (!room) return false;

    if (room.players.length >= room.maxPlayers) {
      return false;
    }

    room.players.push(player);
    return true;
  }

  removePlayer (roomId, playerId) {
    const room = this.getRoom(roomId);
    if (!room) return false;

    room.players = room.players.filter(player => player.id !== playerId);
    return true;
  }

  getPlayers (roomId) {
    const room = this.getRoom(roomId);
    return room ? room.players : [];
  }

  updatePlayerStatus (roomId, playerId, playerIsReady) {
    const room = this.getRoom(roomId);
    if (!room) return false;

    const player = room.players.find(player => player.id === playerId);
    if (!player) return false;

    player.isReady = playerIsReady;
    return true;
  }

  addToPlayerScore (roomId, playerId, amount) {
    const room = this.getRoom(roomId);
    if (!room) return false;

    const player = room.players.find(player => player.id === playerId);
    if (!player) return false;

    player.score += amount;
    return true;
  }

  allReady (roomId) {
    const room = this.getRoom(roomId);
    if (!room || room.players.length === 0) return false;

    return room.players.every(player => player.isReady === true);
  }

  roomExists (id) {
    return this.rooms.some(room => room.id === id);
  }

  updateRoom (roomId, updates) {
    const room = this.getRoom(roomId);
    if (room) {
      Object.assign(room, updates);
    }
    return room;
  }

  resetRoomForNewGame (roomId) {
    const room = this.getRoom(roomId);
    if (!room) return false;

    room.phase = 'lobby';
    room.currentRound = 0;
    room.prompts = [];
    room.currentPrompts = [];
    room.currentPromptIndex = 0;
    room.answers = {};
    room.votes = {};
    room.questions = [];
    room.currentQuestionIndex = 0;
    room.triviaAnswers = {};
    room.drawings = {};
    room.currentDrawingIndex = 0;

    room.players.forEach(player => {
      player.score = 0;
      player.isReady = false;
      player.hasAnswered = false;
      player.hasVoted = false;
      player.hasDrawn = false;
    });

    return room;
  }

  // Clean up old rooms (call periodically)
  cleanupOldRooms (maxAge = 3600000) { // 1 hour default
    const now = Date.now();
    this.rooms = this.rooms.filter(room => now - room.createdAt < maxAge);
  }
}

module.exports = RoomManager;
