<?php
session_start();
require_once 'db_connect.php';

if (!$db_handle) {
    die('Database connection failed');
}

// Get player name from form
$player_name = trim($_POST['player_name'] ?? '');

if (empty($player_name)) {
    header('Location: index.php?error=name_required');
    exit;
}

// Generate unique game code (6 characters, uppercase letters and numbers)
function generateGameCode($db) {
    $characters = 'ABCDEFGHJKLMNPQRSTUVWXYZ23456789'; // Removed confusing chars like 0/O, 1/I
    $attempts = 0;
    
    do {
        $code = '';
        for ($i = 0; $i < 6; $i++) {
            $code .= $characters[random_int(0, strlen($characters) - 1)];
        }
        
        // Check if code already exists
        $stmt = mysqli_prepare($db, "SELECT id FROM razrox_games WHERE game_code = ? AND is_active = 1");
        mysqli_stmt_bind_param($stmt, 's', $code);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $exists = mysqli_num_rows($result) > 0;
        mysqli_stmt_close($stmt);
        
        $attempts++;
    } while ($exists && $attempts < 10);
    
    return $code;
}

// Generate session token for player
function generateSessionToken() {
    return bin2hex(random_bytes(32));
}

// Start transaction
mysqli_begin_transaction($db_handle);

try {
    // Generate game code
    $game_code = generateGameCode($db_handle);
    
    // Create the game
    $stmt = mysqli_prepare($db_handle, "
        INSERT INTO razrox_games (game_code, is_active, game_status) 
        VALUES (?, 1, 'lobby')
    ");
    mysqli_stmt_bind_param($stmt, 's', $game_code);
    mysqli_stmt_execute($stmt);
    $game_id = mysqli_insert_id($db_handle);
    mysqli_stmt_close($stmt);
    
    // Generate session token for host
    $session_token = generateSessionToken();
    
    // Create the host player
    $stmt = mysqli_prepare($db_handle, "
        INSERT INTO razrox_players (game_id, player_name, is_host, session_token) 
        VALUES (?, ?, 1, ?)
    ");
    mysqli_stmt_bind_param($stmt, 'iss', $game_id, $player_name, $session_token);
    mysqli_stmt_execute($stmt);
    $player_id = mysqli_insert_id($db_handle);
    mysqli_stmt_close($stmt);
    
    // Update game with host player id
    $stmt = mysqli_prepare($db_handle, "UPDATE razrox_games SET host_player_id = ? WHERE id = ?");
    mysqli_stmt_bind_param($stmt, 'ii', $player_id, $game_id);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    
    mysqli_commit($db_handle);
    
    // Store session info
    $_SESSION['razrox_game_id'] = $game_id;
    $_SESSION['razrox_player_id'] = $player_id;
    $_SESSION['razrox_session_token'] = $session_token;
    $_SESSION['razrox_is_host'] = true;
    
    // Redirect to lobby
    header('Location: lobby.php?code=' . $game_code);
    exit;
    
} catch (Exception $e) {
    mysqli_rollback($db_handle);
    header('Location: index.php?error=create_failed');
    exit;
}
