<?php
session_start();
require_once 'db_connect.php';

if (!$db_handle) {
    die('Database connection failed');
}

// Verify session
$game_id = $_SESSION['razrox_game_id'] ?? null;
$player_id = $_SESSION['razrox_player_id'] ?? null;
$session_token = $_SESSION['razrox_session_token'] ?? null;
$is_host = $_SESSION['razrox_is_host'] ?? false;

if (!$game_id || !$player_id || !$session_token) {
    header('Location: index.php');
    exit;
}

// Verify player session is valid
$stmt = mysqli_prepare($db_handle, "
    SELECT p.*, g.game_code, g.game_status, g.is_active
    FROM razrox_players p
    JOIN razrox_games g ON g.id = p.game_id
    WHERE p.id = ? AND p.session_token = ? AND g.id = ?
");
mysqli_stmt_bind_param($stmt, 'isi', $player_id, $session_token, $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$player = mysqli_fetch_assoc($result);
mysqli_stmt_close($stmt);

if (!$player || !$player['is_active']) {
    session_destroy();
    header('Location: index.php?error=session_invalid');
    exit;
}

$game_code = $player['game_code'];

// Update player's last activity
$stmt = mysqli_prepare($db_handle, "UPDATE razrox_players SET last_activity = NOW(), is_connected = 1 WHERE id = ?");
mysqli_stmt_bind_param($stmt, 'i', $player_id);
mysqli_stmt_execute($stmt);
mysqli_stmt_close($stmt);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>RazRox - Game Lobby</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        
        .lobby-container {
            max-width: 800px;
            margin: 0 auto;
        }
        
        .header {
            text-align: center;
            margin-bottom: 2rem;
        }
        
        .logo {
            font-size: 2.5rem;
            font-weight: 900;
            color: white;
            text-shadow: 3px 3px 0 rgba(0,0,0,0.2);
        }
        
        .game-code-display {
            background: white;
            border-radius: 20px;
            padding: 2rem;
            text-align: center;
            margin-bottom: 2rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        
        .game-code-label {
            font-size: 1rem;
            color: #666;
            margin-bottom: 0.5rem;
            text-transform: uppercase;
            letter-spacing: 2px;
        }
        
        .game-code {
            font-size: 4rem;
            font-weight: 900;
            color: #667eea;
            letter-spacing: 10px;
            text-shadow: 2px 2px 0 rgba(0,0,0,0.1);
        }
        
        .share-hint {
            margin-top: 1rem;
            color: #888;
            font-size: 0.95rem;
        }
        
        .players-card {
            background: white;
            border-radius: 20px;
            padding: 2rem;
            box-shadow: 0 10px 40px rgba(0,0,0,0.2);
        }
        
        .players-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }
        
        .players-title {
            font-size: 1.5rem;
            font-weight: 700;
            color: #333;
        }
        
        .player-count {
            background: #667eea;
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .players-list {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(180px, 1fr));
            gap: 1rem;
            margin-bottom: 2rem;
        }
        
        .player-card {
            background: linear-gradient(135deg, #f5f7fa 0%, #e4e8eb 100%);
            border-radius: 12px;
            padding: 1rem;
            text-align: center;
            transition: transform 0.2s;
        }
        
        .player-card.host {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .player-card.current {
            border: 3px solid #f5576c;
        }
        
        .player-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            background: #ddd;
            margin: 0 auto 0.8rem;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 1.5rem;
            font-weight: bold;
        }
        
        .player-card.host .player-avatar {
            background: rgba(255,255,255,0.3);
        }
        
        .player-name {
            font-weight: 600;
            font-size: 1.1rem;
        }
        
        .player-badge {
            font-size: 0.75rem;
            background: rgba(0,0,0,0.1);
            padding: 0.2rem 0.6rem;
            border-radius: 10px;
            margin-top: 0.5rem;
            display: inline-block;
        }
        
        .player-card.host .player-badge {
            background: rgba(255,255,255,0.3);
        }
        
        .waiting-message {
            text-align: center;
            padding: 2rem;
            color: #888;
            font-style: italic;
        }
        
        .actions {
            display: flex;
            gap: 1rem;
            flex-wrap: wrap;
        }
        
        .btn {
            flex: 1;
            min-width: 150px;
            padding: 1rem 2rem;
            font-size: 1.1rem;
            font-weight: 700;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.2s;
            text-transform: uppercase;
            letter-spacing: 1px;
            text-decoration: none;
            text-align: center;
        }
        
        .btn-start {
            background: linear-gradient(135deg, #11998e 0%, #38ef7d 100%);
            color: white;
        }
        
        .btn-start:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(17, 153, 142, 0.4);
        }
        
        .btn-start:disabled {
            background: #ccc;
            cursor: not-allowed;
            transform: none;
            box-shadow: none;
        }
        
        .btn-leave {
            background: #eee;
            color: #666;
        }
        
        .btn-leave:hover {
            background: #ddd;
        }
        
        .status-indicator {
            display: inline-block;
            width: 10px;
            height: 10px;
            background: #38ef7d;
            border-radius: 50%;
            margin-right: 8px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        .refreshing {
            color: #888;
            font-size: 0.9rem;
            text-align: center;
            margin-top: 1rem;
        }
    </style>
</head>
<body>
    <div class="lobby-container">
        <div class="header">
            <h1 class="logo">RazRox</h1>
        </div>
        
        <div class="game-code-display">
            <div class="game-code-label">Game Code</div>
            <div class="game-code"><?= htmlspecialchars($game_code) ?></div>
            <div class="share-hint">Share this code with your friends to join!</div>
        </div>
        
        <div class="players-card">
            <div class="players-header">
                <h2 class="players-title"><span class="status-indicator"></span>Players in Lobby</h2>
                <span class="player-count" id="player-count">Loading...</span>
            </div>
            
            <div class="players-list" id="players-list">
                <div class="waiting-message">Loading players...</div>
            </div>
            
            <div class="actions">
                <?php if ($is_host): ?>
                    <button class="btn btn-start" id="start-btn" disabled onclick="startGame()">
                        Start Game
                    </button>
                <?php else: ?>
                    <div class="waiting-message" style="flex:1; padding: 1rem;">
                        Waiting for host to start the game...
                    </div>
                <?php endif; ?>
                <a href="leave_game.php" class="btn btn-leave">Leave Game</a>
            </div>
            
            <div class="refreshing">
                <span id="refresh-status">Auto-refreshing player list...</span>
            </div>
        </div>
    </div>
    
    <script>
        const currentPlayerId = <?= $player_id ?>;
        const isHost = <?= $is_host ? 'true' : 'false' ?>;
        const gameId = <?= $game_id ?>;
        
        // Color palette for player avatars
        const colors = ['#667eea', '#f5576c', '#11998e', '#fc4a1a', '#7b4397', '#43cea2', '#ff6b6b', '#4ecdc4'];
        
        function getPlayerColor(index) {
            return colors[index % colors.length];
        }
        
        function getInitials(name) {
            return name.split(' ').map(n => n[0]).join('').toUpperCase().substring(0, 2);
        }
        
        function updatePlayersList() {
            fetch('get_players.php?game_id=' + gameId)
                .then(response => response.json())
                .then(data => {
                    if (data.error) {
                        console.error(data.error);
                        return;
                    }
                    
                    const container = document.getElementById('players-list');
                    const countEl = document.getElementById('player-count');
                    
                    if (data.players.length === 0) {
                        container.innerHTML = '<div class="waiting-message">No players yet...</div>';
                        countEl.textContent = '0 players';
                        return;
                    }
                    
                    countEl.textContent = data.players.length + ' player' + (data.players.length !== 1 ? 's' : '');
                    
                    let html = '';
                    data.players.forEach((player, index) => {
                        const isCurrentPlayer = player.id == currentPlayerId;
                        const isPlayerHost = player.is_host == 1;
                        
                        let classes = 'player-card';
                        if (isPlayerHost) classes += ' host';
                        if (isCurrentPlayer) classes += ' current';
                        
                        html += `
                            <div class="${classes}">
                                <div class="player-avatar" style="background: ${isPlayerHost ? 'rgba(255,255,255,0.3)' : getPlayerColor(index)}; color: white;">
                                    ${getInitials(player.player_name)}
                                </div>
                                <div class="player-name">${escapeHtml(player.player_name)}</div>
                                ${isPlayerHost ? '<span class="player-badge">👑 Host</span>' : ''}
                                ${isCurrentPlayer && !isPlayerHost ? '<span class="player-badge">You</span>' : ''}
                            </div>
                        `;
                    });
                    
                    container.innerHTML = html;
                    
                    // Enable start button if host and enough players
                    if (isHost) {
                        const startBtn = document.getElementById('start-btn');
                        if (startBtn) {
                            startBtn.disabled = data.players.length < 2;
                        }
                    }
                    
                    // Check if game has started
                    if (data.game_status === 'playing') {
                        window.location.href = 'game.php';
                    }
                })
                .catch(err => {
                    console.error('Error fetching players:', err);
                });
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        function startGame() {
            if (!isHost) return;
            
            fetch('start_game.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({game_id: gameId})
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    window.location.href = 'game.php';
                } else {
                    alert(data.error || 'Failed to start game');
                }
            })
            .catch(err => {
                alert('Error starting game');
            });
        }
        
        // Initial load
        updatePlayersList();
        
        // Poll for updates every 2 seconds
        setInterval(updatePlayersList, 2000);
        
        // Update last activity to keep connection alive
        setInterval(() => {
            fetch('heartbeat.php');
        }, 10000);
    </script>
</body>
</html>
