<?php
session_start();
require_once 'db_connect.php';

// Verify session
$game_id = $_SESSION['razrox_game_id'] ?? null;
$player_id = $_SESSION['razrox_player_id'] ?? null;
$session_token = $_SESSION['razrox_session_token'] ?? null;
$is_host = $_SESSION['razrox_is_host'] ?? false;

if (!$game_id || !$player_id || !$session_token) {
    header('Location: index.php');
    exit;
}

// Verify player and game
$stmt = mysqli_prepare($db_handle, "
    SELECT p.player_name, p.score, g.game_code, g.game_status, g.current_round, g.total_rounds
    FROM razrox_players p
    JOIN razrox_games g ON g.id = p.game_id
    WHERE p.id = ? AND p.session_token = ? AND g.id = ? AND g.is_active = 1
");
mysqli_stmt_bind_param($stmt, 'isi', $player_id, $session_token, $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$data = mysqli_fetch_assoc($result);
mysqli_stmt_close($stmt);

if (!$data) {
    session_destroy();
    header('Location: index.php?error=session_invalid');
    exit;
}

// Update player activity
$stmt = mysqli_prepare($db_handle, "UPDATE razrox_players SET last_activity = NOW(), is_connected = 1 WHERE id = ?");
mysqli_stmt_bind_param($stmt, 'i', $player_id);
mysqli_stmt_execute($stmt);
mysqli_stmt_close($stmt);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>RazRox Trivia</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            min-height: 100vh;
            color: white;
        }
        
        .game-header {
            background: rgba(0,0,0,0.3);
            padding: 1rem 2rem;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .logo {
            font-size: 1.5rem;
            font-weight: 900;
            color: #667eea;
        }
        
        .game-info {
            display: flex;
            gap: 2rem;
            align-items: center;
        }
        
        .round-display {
            font-size: 1rem;
            color: #aaa;
        }
        
        .score-display {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            padding: 0.5rem 1.5rem;
            border-radius: 20px;
            font-weight: bold;
        }
        
        .main-container {
            max-width: 900px;
            margin: 0 auto;
            padding: 2rem;
        }
        
        /* Timer */
        .timer-container {
            text-align: center;
            margin-bottom: 2rem;
        }
        
        .timer {
            font-size: 5rem;
            font-weight: 900;
            color: #38ef7d;
            text-shadow: 0 0 30px rgba(56, 239, 125, 0.5);
            transition: color 0.3s;
        }
        
        .timer.warning {
            color: #f5576c;
            text-shadow: 0 0 30px rgba(245, 87, 108, 0.5);
            animation: pulse 0.5s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { transform: scale(1); }
            50% { transform: scale(1.05); }
        }
        
        /* Question Card */
        .question-card {
            background: rgba(255,255,255,0.1);
            border-radius: 20px;
            padding: 2rem;
            margin-bottom: 2rem;
            backdrop-filter: blur(10px);
        }
        
        .category {
            color: #667eea;
            font-size: 0.9rem;
            text-transform: uppercase;
            letter-spacing: 2px;
            margin-bottom: 1rem;
        }
        
        .difficulty {
            display: inline-block;
            padding: 0.3rem 0.8rem;
            border-radius: 10px;
            font-size: 0.8rem;
            margin-left: 1rem;
        }
        
        .difficulty.easy { background: #38ef7d; color: #000; }
        .difficulty.medium { background: #f7971e; color: #000; }
        .difficulty.hard { background: #f5576c; color: #fff; }
        
        .question-text {
            font-size: 1.8rem;
            line-height: 1.4;
            margin-bottom: 1rem;
        }
        
        /* Answer Options */
        .answers-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 1rem;
        }
        
        .answer-btn {
            background: rgba(255,255,255,0.1);
            border: 3px solid rgba(255,255,255,0.2);
            border-radius: 15px;
            padding: 1.5rem;
            font-size: 1.2rem;
            color: white;
            cursor: pointer;
            transition: all 0.2s;
            text-align: left;
        }
        
        .answer-btn:hover:not(:disabled) {
            background: rgba(102, 126, 234, 0.3);
            border-color: #667eea;
            transform: translateY(-2px);
        }
        
        .answer-btn:disabled {
            cursor: not-allowed;
            opacity: 0.7;
        }
        
        .answer-btn.selected {
            background: rgba(102, 126, 234, 0.5);
            border-color: #667eea;
        }
        
        .answer-btn.correct {
            background: rgba(56, 239, 125, 0.5);
            border-color: #38ef7d;
        }
        
        .answer-btn.incorrect {
            background: rgba(245, 87, 108, 0.5);
            border-color: #f5576c;
        }
        
        .answer-label {
            display: inline-block;
            width: 30px;
            height: 30px;
            background: rgba(255,255,255,0.2);
            border-radius: 50%;
            text-align: center;
            line-height: 30px;
            margin-right: 1rem;
            font-weight: bold;
        }
        
        /* Waiting/Results States */
        .waiting-screen, .results-screen {
            text-align: center;
            padding: 3rem;
        }
        
        .waiting-screen h2 {
            font-size: 2rem;
            margin-bottom: 1rem;
        }
        
        .spinner {
            width: 60px;
            height: 60px;
            border: 5px solid rgba(255,255,255,0.2);
            border-top-color: #667eea;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 2rem auto;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Results */
        .results-card {
            background: rgba(255,255,255,0.1);
            border-radius: 20px;
            padding: 2rem;
            margin-bottom: 2rem;
        }
        
        .result-header {
            font-size: 1.5rem;
            margin-bottom: 1rem;
        }
        
        .points-earned {
            font-size: 3rem;
            font-weight: 900;
            color: #38ef7d;
            margin: 1rem 0;
        }
        
        .points-breakdown {
            color: #aaa;
            font-size: 0.9rem;
        }
        
        /* Leaderboard */
        .leaderboard {
            background: rgba(255,255,255,0.05);
            border-radius: 15px;
            padding: 1.5rem;
            margin-top: 2rem;
        }
        
        .leaderboard h3 {
            margin-bottom: 1rem;
            color: #667eea;
        }
        
        .leaderboard-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 0.8rem;
            border-radius: 10px;
            margin-bottom: 0.5rem;
        }
        
        .leaderboard-item.current-player {
            background: rgba(102, 126, 234, 0.2);
        }
        
        .leaderboard-item:nth-child(1) { background: rgba(255, 215, 0, 0.2); }
        .leaderboard-item:nth-child(2) { background: rgba(192, 192, 192, 0.2); }
        .leaderboard-item:nth-child(3) { background: rgba(205, 127, 50, 0.2); }
        
        .player-rank {
            font-weight: bold;
            width: 30px;
        }
        
        .player-name {
            flex: 1;
            margin-left: 1rem;
        }
        
        .player-score {
            font-weight: bold;
            color: #38ef7d;
        }
        
        /* Host Controls */
        .host-controls {
            text-align: center;
            margin-top: 2rem;
        }
        
        .btn {
            padding: 1rem 2rem;
            font-size: 1.1rem;
            font-weight: 700;
            border: none;
            border-radius: 12px;
            cursor: pointer;
            transition: all 0.2s;
            margin: 0.5rem;
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 25px rgba(102, 126, 234, 0.4);
        }
        
        .btn-secondary {
            background: rgba(255,255,255,0.1);
            color: white;
        }
        
        /* Game Over */
        .game-over {
            text-align: center;
        }
        
        .game-over h1 {
            font-size: 3rem;
            margin-bottom: 1rem;
            color: #667eea;
        }
        
        .winner-name {
            font-size: 2rem;
            color: #38ef7d;
            margin: 1rem 0;
        }
        
        .final-scores {
            margin: 2rem 0;
        }
        
        @media (max-width: 600px) {
            .answers-grid {
                grid-template-columns: 1fr;
            }
            
            .question-text {
                font-size: 1.3rem;
            }
            
            .timer {
                font-size: 3rem;
            }
        }
    </style>
</head>
<body>
    <header class="game-header">
        <div class="logo">RazRox</div>
        <div class="game-info">
            <span class="round-display">Round <span id="current-round">0</span>/<span id="total-rounds"><?= $data['total_rounds'] ?></span></span>
            <div class="score-display">
                <span id="player-score"><?= $data['score'] ?></span> pts
            </div>
        </div>
    </header>
    
    <div class="main-container">
        <!-- Waiting for round to start -->
        <div id="waiting-screen" class="waiting-screen">
            <h2>Get Ready!</h2>
            <div class="spinner"></div>
            <p id="waiting-message">Waiting for the next question...</p>
        </div>
        
        <!-- Question Screen -->
        <div id="question-screen" style="display: none;">
            <div class="timer-container">
                <div class="timer" id="timer">10.0</div>
            </div>
            
            <div class="question-card">
                <div>
                    <span class="category" id="question-category">Category</span>
                    <span class="difficulty" id="question-difficulty">Easy</span>
                </div>
                <p class="question-text" id="question-text">Loading question...</p>
            </div>
            
            <div class="answers-grid" id="answers-container">
                <!-- Answers will be inserted here -->
            </div>
        </div>
        
        <!-- Results Screen -->
        <div id="results-screen" class="results-screen" style="display: none;">
            <div class="results-card">
                <h2 class="result-header" id="result-header">Time's Up!</h2>
                <p id="correct-answer-display"></p>
                <div class="points-earned" id="points-earned">+0</div>
                <p class="points-breakdown" id="points-breakdown"></p>
            </div>
            
            <div class="leaderboard" id="leaderboard">
                <h3>🏆 Leaderboard</h3>
                <div id="leaderboard-list"></div>
            </div>
            
            <?php if ($is_host): ?>
            <div class="host-controls">
                <button class="btn btn-primary" id="next-round-btn" onclick="nextRound()">Next Question</button>
            </div>
            <?php else: ?>
            <p style="text-align: center; color: #888; margin-top: 2rem;">Waiting for host to continue...</p>
            <?php endif; ?>
        </div>
        
        <!-- Game Over Screen -->
        <div id="gameover-screen" class="game-over" style="display: none;">
            <h1>🎉 Game Over!</h1>
            <p>The winner is...</p>
            <div class="winner-name" id="winner-name">Loading...</div>
            
            <div class="final-scores leaderboard" id="final-leaderboard">
                <h3>Final Scores</h3>
                <div id="final-leaderboard-list"></div>
            </div>
            
            <div style="margin-top: 2rem;">
                <a href="leave_game.php" class="btn btn-secondary">Back to Home</a>
            </div>
        </div>
    </div>
    
    <script>
        const gameId = <?= $game_id ?>;
        const playerId = <?= $player_id ?>;
        const isHost = <?= $is_host ? 'true' : 'false' ?>;
        const totalRounds = <?= $data['total_rounds'] ?>;
        
        let currentRound = 0;
        let timerInterval = null;
        let timeRemaining = 10000; // milliseconds
        let hasAnswered = false;
        let currentQuestion = null;
        let pollInterval = null;
        
        // Screens
        const waitingScreen = document.getElementById('waiting-screen');
        const questionScreen = document.getElementById('question-screen');
        const resultsScreen = document.getElementById('results-screen');
        const gameoverScreen = document.getElementById('gameover-screen');
        
        function showScreen(screen) {
            waitingScreen.style.display = 'none';
            questionScreen.style.display = 'none';
            resultsScreen.style.display = 'none';
            gameoverScreen.style.display = 'none';
            screen.style.display = 'block';
        }
        
        function updateTimer() {
            timeRemaining -= 100;
            const seconds = (timeRemaining / 1000).toFixed(1);
            const timerEl = document.getElementById('timer');
            timerEl.textContent = seconds;
            
            if (timeRemaining <= 3000) {
                timerEl.classList.add('warning');
            }
            
            if (timeRemaining <= 0) {
                console.log('Timer expired, submitting empty answer');
                clearInterval(timerInterval);
                if (!hasAnswered) {
                    submitAnswer('');
                }
            }
        }
        
        function startTimer(serverTimeRemaining) {
            // Clear any existing timer first
            if (timerInterval) {
                clearInterval(timerInterval);
            }
            
            // Use server's time remaining (in seconds), convert to ms
            // Default to 10 seconds if not provided
            timeRemaining = (serverTimeRemaining || 10) * 1000;
            hasAnswered = false;
            console.log('Starting timer with', timeRemaining, 'ms');
            document.getElementById('timer').classList.remove('warning');
            document.getElementById('timer').textContent = (timeRemaining / 1000).toFixed(1);
            timerInterval = setInterval(updateTimer, 100);
        }
        
        function displayQuestion(data) {
            console.log('displayQuestion called, round:', data.round_number, 'currentRound:', currentRound, 'time_remaining:', data.time_remaining);
            currentQuestion = data;
            currentRound = data.round_number;
            
            document.getElementById('current-round').textContent = currentRound;
            document.getElementById('question-category').textContent = data.category;
            document.getElementById('question-difficulty').textContent = data.difficulty;
            document.getElementById('question-difficulty').className = 'difficulty ' + data.difficulty;
            document.getElementById('question-text').textContent = data.question;
            
            // Shuffle and display answers
            const answers = [
                {label: 'A', text: data.a1},
                {label: 'B', text: data.a2},
                {label: 'C', text: data.a3},
                {label: 'D', text: data.a4}
            ];
            
            const container = document.getElementById('answers-container');
            container.innerHTML = '';
            
            answers.forEach(answer => {
                const btn = document.createElement('button');
                btn.className = 'answer-btn';
                btn.innerHTML = `<span class="answer-label">${answer.label}</span>${escapeHtml(answer.text)}`;
                btn.onclick = () => selectAnswer(answer.text, btn);
                container.appendChild(btn);
            });
            
            showScreen(questionScreen);
            // Pass server's time_remaining (in seconds)
            startTimer(data.time_remaining);
        }
        
        function selectAnswer(answer, btn) {
            if (hasAnswered) return;
            
            hasAnswered = true;
            clearInterval(timerInterval);
            
            // Disable all buttons and highlight selected
            document.querySelectorAll('.answer-btn').forEach(b => {
                b.disabled = true;
            });
            btn.classList.add('selected');
            
            submitAnswer(answer);
        }
        
        function submitAnswer(answer) {
            fetch('trivia_answer.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({
                    game_id: gameId,
                    answer: answer,
                    time_remaining: Math.max(0, timeRemaining)
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.error) {
                    console.error(data.error);
                    return;
                }
                // Wait for results to be shown by polling
            })
            .catch(err => console.error('Error submitting answer:', err));
        }
        
        function showResults(data) {
            const isCorrect = data.your_answer === data.correct_answer && data.your_answer !== '';
            
            // Show correct/incorrect on answer buttons
            document.querySelectorAll('.answer-btn').forEach(btn => {
                const answerText = btn.textContent.substring(1).trim();
                if (answerText === data.correct_answer) {
                    btn.classList.add('correct');
                } else if (answerText === data.your_answer && !isCorrect) {
                    btn.classList.add('incorrect');
                }
            });
            
            // Update results screen
            setTimeout(() => {
                document.getElementById('result-header').textContent = isCorrect ? '✅ Correct!' : '❌ ' + (data.your_answer ? 'Wrong!' : "Time's Up!");
                document.getElementById('correct-answer-display').textContent = 'The answer was: ' + data.correct_answer;
                document.getElementById('points-earned').textContent = '+' + data.points_awarded;
                
                if (data.points_awarded > 0) {
                    const basePoints = 100;
                    const bonusPoints = data.points_awarded - basePoints;
                    document.getElementById('points-breakdown').textContent = 
                        `${basePoints} base + ${bonusPoints} speed bonus`;
                } else {
                    document.getElementById('points-breakdown').textContent = '';
                }
                
                // Update score display
                document.getElementById('player-score').textContent = data.new_score;
                
                // Show leaderboard
                displayLeaderboard(data.leaderboard);
                
                showScreen(resultsScreen);
            }, 1500);
        }
        
        function displayLeaderboard(players) {
            const list = document.getElementById('leaderboard-list');
            list.innerHTML = '';
            
            players.sort((a, b) => b.score - a.score);
            
            players.forEach((player, index) => {
                const item = document.createElement('div');
                item.className = 'leaderboard-item' + (player.id == playerId ? ' current-player' : '');
                item.innerHTML = `
                    <span class="player-rank">${index + 1}.</span>
                    <span class="player-name">${escapeHtml(player.player_name)}</span>
                    <span class="player-score">${player.score} pts</span>
                `;
                list.appendChild(item);
            });
        }
        
        function nextRound() {
            if (!isHost) return;
            
            document.getElementById('next-round-btn').disabled = true;
            
            fetch('trivia_next.php', {
                method: 'POST',
                headers: {'Content-Type': 'application/json'},
                body: JSON.stringify({game_id: gameId})
            })
            .then(response => response.json())
            .then(data => {
                document.getElementById('next-round-btn').disabled = false;
                if (data.error) {
                    alert(data.error);
                } else {
                    // Reset hasAnswered for new round and immediately poll
                    hasAnswered = false;
                    console.log('Round started, polling immediately. currentRound:', currentRound);
                    pollGameState();
                }
            })
            .catch(err => {
                document.getElementById('next-round-btn').disabled = false;
                console.error('Error starting next round:', err);
            });
        }
        
        function showGameOver(data) {
            displayFinalLeaderboard(data.leaderboard);
            
            if (data.leaderboard.length > 0) {
                const winner = data.leaderboard.sort((a, b) => b.score - a.score)[0];
                document.getElementById('winner-name').textContent = '🏆 ' + winner.player_name + ' 🏆';
            }
            
            showScreen(gameoverScreen);
        }
        
        function displayFinalLeaderboard(players) {
            const list = document.getElementById('final-leaderboard-list');
            list.innerHTML = '';
            
            players.sort((a, b) => b.score - a.score);
            
            players.forEach((player, index) => {
                const medal = index === 0 ? '🥇' : (index === 1 ? '🥈' : (index === 2 ? '🥉' : ''));
                const item = document.createElement('div');
                item.className = 'leaderboard-item' + (player.id == playerId ? ' current-player' : '');
                item.innerHTML = `
                    <span class="player-rank">${medal || (index + 1) + '.'}</span>
                    <span class="player-name">${escapeHtml(player.player_name)}</span>
                    <span class="player-score">${player.score} pts</span>
                `;
                list.appendChild(item);
            });
        }
        
        function pollGameState() {
            fetch('trivia_state.php?game_id=' + gameId)
                .then(response => response.json())
                .then(data => {
                    console.log('Poll response:', data);
                    
                    if (data.error) {
                        console.error(data.error);
                        return;
                    }
                    
                    if (data.game_status === 'finished') {
                        clearInterval(pollInterval);
                        showGameOver(data);
                        return;
                    }
                    
                    if (data.round_active && data.question && !hasAnswered && data.round_number > currentRound) {
                        console.log('Condition met! Showing question. hasAnswered:', hasAnswered, 'data.round_number:', data.round_number, 'currentRound:', currentRound);
                        displayQuestion(data.question);
                    } else if (!data.round_active && data.round_number > 0 && data.round_number >= currentRound) {
                        console.log('Showing results. round_active:', data.round_active, 'round_number:', data.round_number, 'currentRound:', currentRound);
                        if (data.results) {
                            showResults(data.results);
                        }
                    } else if (!data.round_active && data.round_number === 0) {
                        console.log('Showing waiting screen');
                        showScreen(waitingScreen);
                        if (isHost) {
                            document.getElementById('waiting-message').innerHTML = 
                                '<button class="btn btn-primary" onclick="nextRound()">Start First Question</button>';
                        } else {
                            document.getElementById('waiting-message').textContent = 'Waiting for host to start...';
                        }
                    }
                })
                .catch(err => console.error('Error polling state:', err));
        }
        
        function escapeHtml(text) {
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Keep connection alive
        setInterval(() => fetch('heartbeat.php'), 10000);
        
        // If host, show start button initially before polling starts
        <?php if ($is_host): ?>
        document.getElementById('waiting-message').innerHTML = 
            '<button class="btn btn-primary" onclick="nextRound()">Start First Question</button>';
        <?php endif; ?>
        
        // Start polling
        pollGameState();
        pollInterval = setInterval(pollGameState, 1000);
    </script>
</body>
</html>
