<?php
session_start();
require_once 'db_connect.php';

header('Content-Type: application/json');

$data = json_decode(file_get_contents('php://input'), true);

if (!isset($data['game_id']) || !isset($data['answer']) || !isset($data['time_remaining'])) {
    echo json_encode(['error' => 'Missing required fields']);
    exit;
}

$game_id = (int)$data['game_id'];
$answer = $data['answer'];
$time_remaining = (int)$data['time_remaining'];
$player_id = (int)$_SESSION['player_id'];

// Get current active round
$stmt = mysqli_prepare($db_handle, "
    SELECT r.id, t.acorrect
    FROM razrox_trivia_rounds r
    JOIN Trivia t ON r.question_id = t.id
    WHERE r.game_id = ? AND r.is_active = 1
");
mysqli_stmt_bind_param($stmt, "i", $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$round = mysqli_fetch_assoc($result);

if (!$round) {
    echo json_encode(['error' => 'No active round']);
    exit;
}

// Check if player already answered
$stmt = mysqli_prepare($db_handle, "SELECT id FROM razrox_trivia_answers WHERE round_id = ? AND player_id = ?");
mysqli_stmt_bind_param($stmt, "ii", $round['id'], $player_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
if (mysqli_fetch_assoc($result)) {
    echo json_encode(['error' => 'Already answered this round']);
    exit;
}

// Calculate points
$is_correct = ($answer === $round['acorrect']) ? 1 : 0;
$points = 0;
if ($is_correct && $answer !== '') {
    // Base 100 points + 1 point per 100ms remaining
    $points = 100 + floor($time_remaining / 100);
}

// Start transaction
mysqli_begin_transaction($db_handle);

try {
    // Record answer
    $stmt = mysqli_prepare($db_handle, "
        INSERT INTO razrox_trivia_answers (round_id, player_id, answer, is_correct, time_remaining_ms, points_awarded)
        VALUES (?, ?, ?, ?, ?, ?)
    ");
    mysqli_stmt_bind_param($stmt, "iisiii", $round['id'], $player_id, $answer, $is_correct, $time_remaining, $points);
    mysqli_stmt_execute($stmt);
    
    // Update player score
    $stmt = mysqli_prepare($db_handle, "UPDATE razrox_players SET score = score + ? WHERE id = ?");
    mysqli_stmt_bind_param($stmt, "ii", $points, $player_id);
    mysqli_stmt_execute($stmt);
    
    mysqli_commit($db_handle);
    
    echo json_encode([
        'success' => true,
        'points_earned' => $points,
        'is_correct' => (bool)$is_correct
    ]);
    
} catch (Exception $e) {
    mysqli_rollback($db_handle);
    echo json_encode(['error' => 'Failed to record answer']);
}
