<?php
session_start();
require_once 'db_connect.php';

header('Content-Type: application/json');

$data = json_decode(file_get_contents('php://input'), true);

if (!isset($data['game_id'])) {
    echo json_encode(['error' => 'Game ID required']);
    exit;
}

$game_id = (int)$data['game_id'];
$player_id = (int)$_SESSION['razrox_player_id'];

// Verify player is host
$stmt = mysqli_prepare($db_handle, "SELECT is_host FROM razrox_players WHERE id = ? AND game_id = ?");
mysqli_stmt_bind_param($stmt, "ii", $player_id, $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$player = mysqli_fetch_assoc($result);

if (!$player || !$player['is_host']) {
    echo json_encode(['error' => 'Only host can start rounds']);
    exit;
}

// Get current game state
$stmt = mysqli_prepare($db_handle, "SELECT current_round, total_rounds, game_status FROM razrox_games WHERE id = ?");
mysqli_stmt_bind_param($stmt, "i", $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$game = mysqli_fetch_assoc($result);

if (!$game) {
    echo json_encode(['error' => 'Game not found']);
    exit;
}

if ($game['game_status'] !== 'playing') {
    echo json_encode(['error' => 'Game is not active. Current status: ' . $game['game_status']]);
    exit;
}

$next_round = (int)$game['current_round'] + 1;

if ($next_round > (int)$game['total_rounds']) {
    // Game is complete
    mysqli_query($db_handle, "UPDATE razrox_games SET game_status = 'finished' WHERE id = $game_id");
    echo json_encode(['success' => true, 'game_finished' => true]);
    exit;
}

// End current round if it exists
mysqli_query($db_handle, "
    UPDATE razrox_trivia_rounds 
    SET is_active = 0, ended_at = NOW() 
    WHERE game_id = $game_id AND is_active = 1
");

// Get list of already used question IDs in this game
$stmt = mysqli_prepare($db_handle, "SELECT question_id FROM razrox_trivia_rounds WHERE game_id = ?");
mysqli_stmt_bind_param($stmt, "i", $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$used_questions = [];
while ($row = mysqli_fetch_assoc($result)) {
    $used_questions[] = (int)$row['question_id'];
}

// Get a random question that hasn't been used
$used_ids_str = empty($used_questions) ? '' : implode(',', $used_questions);
$where_clause = empty($used_questions) ? '' : "WHERE id NOT IN ($used_ids_str)";
$query = "SELECT id FROM Trivia $where_clause ORDER BY RAND() LIMIT 1";
$result = mysqli_query($db_handle, $query);
$question = mysqli_fetch_assoc($result);

if (!$question) {
    echo json_encode(['error' => 'No more questions available']);
    exit;
}

// Start transaction
mysqli_begin_transaction($db_handle);

try {
    // Create new round
    $stmt = mysqli_prepare($db_handle, "
        INSERT INTO razrox_trivia_rounds (game_id, question_id, round_number, started_at, is_active)
        VALUES (?, ?, ?, NOW(), 1)
    ");
    mysqli_stmt_bind_param($stmt, "iii", $game_id, $question['id'], $next_round);
    mysqli_stmt_execute($stmt);
    
    // Update game
    $stmt = mysqli_prepare($db_handle, "UPDATE razrox_games SET current_round = ? WHERE id = ?");
    mysqli_stmt_bind_param($stmt, "ii", $next_round, $game_id);
    mysqli_stmt_execute($stmt);
    
    mysqli_commit($db_handle);
    
    echo json_encode([
        'success' => true,
        'round' => $next_round
    ]);
    
} catch (Exception $e) {
    mysqli_rollback($db_handle);
    echo json_encode(['error' => 'Failed to start round']);
}
