<?php
session_start();
require_once 'db_connect.php';

header('Content-Type: application/json');

if (!isset($_GET['game_id'])) {
    echo json_encode(['error' => 'Game ID required']);
    exit;
}

$game_id = (int)$_GET['game_id'];

// Get game status
$stmt = mysqli_prepare($db_handle, "SELECT game_status, current_round, total_rounds FROM razrox_games WHERE id = ?");
mysqli_stmt_bind_param($stmt, "i", $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$game = mysqli_fetch_assoc($result);

if (!$game) {
    echo json_encode(['error' => 'Game not found']);
    exit;
}

$response = [
    'game_status' => $game['game_status'],
    'current_round' => (int)$game['current_round'],
    'total_rounds' => (int)$game['total_rounds'],
    'round_active' => false,
    'round_number' => (int)$game['current_round']
];

// If game is finished, get final leaderboard
if ($game['game_status'] === 'finished') {
    $stmt = mysqli_prepare($db_handle, "SELECT id, player_name, score FROM razrox_players WHERE game_id = ? ORDER BY score DESC");
    mysqli_stmt_bind_param($stmt, "i", $game_id);
    mysqli_stmt_execute($stmt);
    $result = mysqli_stmt_get_result($stmt);
    $response['leaderboard'] = mysqli_fetch_all($result, MYSQLI_ASSOC);
    echo json_encode($response);
    exit;
}

// Check for active round - calculate elapsed time in the database to avoid timezone issues
$stmt = mysqli_prepare($db_handle, "
    SELECT r.id, r.question_id, r.round_number, r.started_at, r.ended_at,
           t.category, t.difficulty, t.question, t.a1, t.a2, t.a3, t.a4, t.acorrect,
           TIMESTAMPDIFF(SECOND, r.started_at, NOW()) as elapsed_seconds
    FROM razrox_trivia_rounds r
    JOIN Trivia t ON r.question_id = t.id
    WHERE r.game_id = ? AND r.is_active = 1
    ORDER BY r.round_number DESC
    LIMIT 1
");
mysqli_stmt_bind_param($stmt, "i", $game_id);
mysqli_stmt_execute($stmt);
$result = mysqli_stmt_get_result($stmt);
$round = mysqli_fetch_assoc($result);

if ($round) {
    $elapsed = (int)$round['elapsed_seconds'];
    
    // Debug info
    $response['debug'] = [
        'elapsed_seconds' => $elapsed,
        'started_at' => $round['started_at'],
        'ended_at' => $round['ended_at']
    ];
    
    // Round is active if started less than 10 seconds ago and not ended
    if ($elapsed < 10 && !$round['ended_at']) {
        $response['round_active'] = true;
        $response['round_number'] = (int)$round['round_number'];
        $response['time_remaining'] = 10 - $elapsed;
        $response['question'] = [
            'round_number' => (int)$round['round_number'],
            'category' => $round['category'],
            'difficulty' => $round['difficulty'],
            'question' => $round['question'],
            'a1' => $round['a1'],
            'a2' => $round['a2'],
            'a3' => $round['a3'],
            'a4' => $round['a4'],
            'time_remaining' => 10 - $elapsed
        ];
    } else {
        // Round ended, show results
        $response['round_active'] = false;
        $response['round_number'] = (int)$round['round_number'];
        
        // Get player's answer
        $player_id = (int)$_SESSION['razrox_player_id'];
        $stmt = mysqli_prepare($db_handle, "
            SELECT answer, is_correct, points_awarded
            FROM razrox_trivia_answers
            WHERE round_id = ? AND player_id = ?
        ");
        mysqli_stmt_bind_param($stmt, "ii", $round['id'], $player_id);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $answer = mysqli_fetch_assoc($result);
        
        $response['results'] = [
            'correct_answer' => $round['acorrect'],
            'your_answer' => $answer ? $answer['answer'] : '',
            'points_earned' => $answer ? (int)$answer['points_awarded'] : 0
        ];
        
        // Get leaderboard
        $stmt = mysqli_prepare($db_handle, "SELECT id, player_name, score FROM razrox_players WHERE game_id = ? ORDER BY score DESC");
        mysqli_stmt_bind_param($stmt, "i", $game_id);
        mysqli_stmt_execute($stmt);
        $result = mysqli_stmt_get_result($stmt);
        $response['results']['leaderboard'] = mysqli_fetch_all($result, MYSQLI_ASSOC);
    }
}

echo json_encode($response);
